import argparse
from datetime import datetime, timedelta
from zoneinfo import ZoneInfo

import requests

WIB = ZoneInfo("Asia/Jakarta")


def iso(dt):
    return dt.astimezone(WIB).isoformat()


def resolve_shift_range(now, shift_key):
    shift_start_hours = {
        "shift1": 7,
        "shift2": 15,
        "shift3": 23,
    }
    if shift_key not in shift_start_hours:
        raise ValueError(f"Unknown shift key: {shift_key}")

    start = now.replace(
        hour=shift_start_hours[shift_key], minute=0, second=0, microsecond=0
    )
    if now < start:
        start -= timedelta(days=1)

    end = start + timedelta(hours=8) - timedelta(minutes=1)
    if end > now:
        end = now

    return start, end


def distribute_times(start, end, count):
    if count <= 0:
        return []
    if end <= start:
        return [start]

    total_seconds = (end - start).total_seconds()
    times = []
    for i in range(count):
        offset = total_seconds * (i + 1) / (count + 1)
        times.append(start + timedelta(seconds=offset))
    return times


def build_payloads(now, shifts, count_per_shift, include_boundaries, prefix):
    payloads = []
    meta = {}

    base_metrics = {
        "shift1": (12.0, 1200.0, 60.0, 20.0),
        "shift2": (13.0, 1400.0, 61.0, 21.0),
        "shift3": (11.0, 1600.0, 59.0, 19.0),
    }

    for shift_key in shifts:
        start, end = resolve_shift_range(now, shift_key)
        meta[shift_key] = (start, end)

        times = []
        if include_boundaries:
            times.extend([start, end])
        times.extend(distribute_times(start, end, count_per_shift))

        seen = set()
        unique_times = []
        for ts in times:
            key = ts.isoformat()
            if key in seen:
                continue
            seen.add(key)
            unique_times.append(ts)

        amp, hm, temp, moist = base_metrics[shift_key]
        for idx, ts in enumerate(unique_times, start=1):
            machine_id = f"{prefix}-{shift_key.upper()}-{idx:02d}"
            payloads.append(
                {
                    "machine_id": machine_id,
                    "amp": float(amp + idx * 0.1),
                    "hm": float(hm + idx),
                    "temp": float(temp + idx * 0.1),
                    "moist": float(moist + idx * 0.1),
                    "timestamp": iso(ts),
                }
            )

    return payloads, meta


def post_payloads(url, payloads, timeout):
    ok = 0
    fail = 0
    session = requests.Session()
    for payload in payloads:
        try:
            resp = session.post(url, json=payload, timeout=timeout)
            if 200 <= resp.status_code < 300:
                ok += 1
            else:
                fail += 1
                print(f"FAIL {payload['machine_id']} ({resp.status_code})")
        except requests.RequestException as exc:
            fail += 1
            print(f"ERROR {payload['machine_id']} ({exc})")
    return ok, fail


def main():
    parser = argparse.ArgumentParser(
        description="Send shift-based machine_readings to test last-shift filters"
    )
    parser.add_argument(
        "--base-url",
        default="http://localhost:8000/api/machine-readings",
        help="API endpoint for machine readings",
    )
    parser.add_argument(
        "--shift",
        default="all",
        choices=["all", "shift1", "shift2", "shift3"],
        help="send data for one shift or all",
    )
    parser.add_argument(
        "--count",
        type=int,
        default=1,
        help="extra points per shift between boundaries",
    )
    parser.add_argument(
        "--no-boundaries",
        action="store_true",
        help="skip exact start/end timestamps",
    )
    parser.add_argument(
        "--prefix",
        default="SHIFT",
        help="machine_id prefix",
    )
    parser.add_argument(
        "--timeout",
        type=int,
        default=5,
        help="request timeout in seconds",
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="print payloads without sending",
    )
    args = parser.parse_args()

    now = datetime.now(WIB)
    if args.shift == "all":
        shifts = ["shift1", "shift2", "shift3"]
    else:
        shifts = [args.shift]

    payloads, meta = build_payloads(
        now,
        shifts,
        count_per_shift=args.count,
        include_boundaries=not args.no_boundaries,
        prefix=args.prefix,
    )

    if args.dry_run:
        for payload in payloads:
            print(payload)
        return

    ok, fail = post_payloads(args.base_url, payloads, args.timeout)
    print(f"Shift seed complete. OK={ok}, FAIL={fail}")
    print("Shift ranges (WIB):")
    for shift_key in shifts:
        start, end = meta[shift_key]
        print(f"- {shift_key}: {iso(start)} to {iso(end)}")


if __name__ == "__main__":
    main()
