import argparse
import random
import time
from datetime import datetime, timedelta
from zoneinfo import ZoneInfo

import requests

WIB = ZoneInfo("Asia/Jakarta")


def iso(dt):
    return dt.astimezone(WIB).isoformat()


def make_payload(machine_id, amp, hm, temp, moist, ts):
    return {
        "machine_id": machine_id,
        "amp": float(amp),
        "hm": float(hm),
        "temp": float(temp),
        "moist": float(moist),
        "timestamp": ts,
    }


def build_cases(now):
    recent_time = now - timedelta(minutes=10)
    recent_time2 = now - timedelta(minutes=5)
    custom_time = now - timedelta(days=2, minutes=10)
    old_time = now - timedelta(days=10)

    cases = []

    def add(machine_id, amp, hm, temp, moist, ts):
        cases.append(make_payload(machine_id, amp, hm, temp, moist, iso(ts)))

    # HM priority (ASC)
    add("HM-100", 16, 100, 60, 20, recent_time)
    add("HM-500", 16, 500, 60, 20, recent_time)
    add("HM-900", 16, 900, 60, 20, recent_time)

    # Amp priority within same HM (HM=500)
    add("AMP-10", 10, 500, 60, 20, recent_time)
    add("AMP-13", 13, 500, 60, 20, recent_time)
    add("AMP-16", 16, 500, 60, 20, recent_time)

    # Temp priority within same HM and Amp (HM=700, Amp=16)
    for v in [80, 75, 70, 50, 58, 62, 59, 60]:
        add(f"TEMP-{v}", 16, 700, v, 20, recent_time)

    # Moist priority within same HM, Amp, Temp (HM=800, Amp=16, Temp=60)
    for v in [5, 10, 15, 20, 25, 30]:
        add(f"MOIST-{v:02d}", 16, 800, 60, v, recent_time)

    # Average-based cases (multiple readings per machine)
    add("AVG-AMP-13_5", 10, 600, 60, 20, recent_time)
    add("AVG-AMP-13_5", 17, 600, 60, 20, recent_time2)

    add("AVG-TEMP-62", 16, 650, 60, 20, recent_time)
    add("AVG-TEMP-62", 16, 650, 64, 20, recent_time2)

    add("AVG-MOIST-23", 16, 650, 60, 20, recent_time)
    add("AVG-MOIST-23", 16, 650, 60, 26, recent_time2)

    # Custom range only (older timestamp)
    add("CUSTOM-ONLY", 12, 3000, 80, 30, custom_time)

    # Old-only machine (should still appear in grid/latest)
    add("OLD-ONLY", 18, 2000, 55, 10, old_time)

    meta = {
        "custom_start": iso(custom_time - timedelta(minutes=5)),
        "custom_end": iso(custom_time + timedelta(minutes=5)),
        "recent_start": iso(recent_time - timedelta(minutes=30)),
        "recent_end": iso(now + timedelta(minutes=1)),
    }

    return cases, meta


def post_cases(url, cases):
    ok = 0
    fail = 0
    session = requests.Session()
    for payload in cases:
        try:
            resp = session.post(url, json=payload, timeout=5)
            if resp.status_code == 201:
                ok += 1
            else:
                fail += 1
                print(f"FAIL {payload['machine_id']} ({resp.status_code})")
        except requests.RequestException as exc:
            fail += 1
            print(f"ERROR {payload['machine_id']} ({exc})")
    return ok, fail


def stream_updates(url, machine_ids, interval_seconds):
    session = requests.Session()
    print("Streaming random updates. Press Ctrl+C to stop.")
    try:
        while True:
            now = datetime.now(WIB)
            for mid in machine_ids:
                payload = make_payload(
                    mid,
                    random.uniform(5.0, 20.0),
                    random.uniform(1000.0, 5000.0),
                    random.uniform(40.0, 80.0),
                    random.uniform(5.0, 50.0),
                    iso(now),
                )
                try:
                    session.post(url, json=payload, timeout=5)
                except requests.RequestException:
                    pass
            time.sleep(interval_seconds)
    except KeyboardInterrupt:
        print("Stream stopped.")


def main():
    parser = argparse.ArgumentParser(description="Seed machine_readings for sorting checks")
    parser.add_argument("--base-url", default="https://iot4.tujukuda.com/api/machine-readings")
    parser.add_argument("--stream", action="store_true", help="send random updates after seeding")
    parser.add_argument("--interval", type=int, default=5, help="seconds between stream batches")
    args = parser.parse_args()

    now = datetime.now(WIB)
    cases, meta = build_cases(now)
    ok, fail = post_cases(args.base_url, cases)

    print(f"Seed complete. OK={ok}, FAIL={fail}")
    print("Suggested checks:")
    print(f"- Custom range: {meta['custom_start']} to {meta['custom_end']}")
    print(f"- Recent range: {meta['recent_start']} to {meta['recent_end']}")
    print("- Grid/latest should include OLD-ONLY even if not in recent range")
    print("- HM order example (HM-100, HM-500, HM-900)")
    print("- Amp order within HM=500: AMP-10, AMP-13, AVG-AMP-13_5, AMP-16")
    print("- Temp order within HM=700 (dev desc, value asc): TEMP-80, TEMP-75, TEMP-50, TEMP-70, TEMP-58, TEMP-62, TEMP-59, TEMP-60")
    print("- Moist order within HM=800 (dev desc, value asc): MOIST-05, MOIST-10, MOIST-30, MOIST-15, MOIST-25, MOIST-20")

    if args.stream:
        stream_ids = ["HM-100", "AMP-10", "TEMP-80", "MOIST-05"]
        stream_updates(args.base_url, stream_ids, args.interval)


if __name__ == "__main__":
    main()
